#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Gerador de gráficos como imagens usando matplotlib
"""
import os
import base64
import io
import matplotlib.pyplot as plt
import matplotlib
matplotlib.use('Agg')  # Backend sem GUI

# Importa sistema de tradução
try:
    from .i18n import t
except ImportError:
    from i18n import t

def generate_chart_image(times, values, test_name, max_time=None):
    """
    Gera uma imagem do gráfico usando matplotlib
    
    Args:
        times: Lista de tempos
        values: Lista de valores
        test_name: Nome do teste
        max_time: Tempo máximo (opcional)
        
    Returns:
        String base64 da imagem PNG
    """
    if not times or not values or len(times) != len(values):
        return None
    
    # Configuração da figura - formato fixo que cabe no PDF
    # Sempre as mesmas dimensões que cabem na página
    plt.figure(figsize=(12, 4))  # Dimensões fixas que cabem no PDF
    plt.style.use('default')
    
    # Plota o gráfico
    plt.plot(times, values, 'o-', color='#8B4513', linewidth=2, markersize=4, markerfacecolor='#8B4513', markeredgecolor='white', markeredgewidth=1)
    
    # Configurações do gráfico
    plt.title(f'{t("pdf.noise_chart_title")} - {test_name}', fontsize=14, fontweight='bold', color='#2c3e50')
    plt.xlabel(t('noise.x_axis_label'), fontsize=12)
    plt.ylabel(t('noise.y_axis_label'), fontsize=12)
    
    # Configurações dos eixos
    plt.grid(True, alpha=0.3, linestyle='--')
    plt.ylim(-80, -40)
    
    # Eixo X se adapta automaticamente aos dados dentro do formato fixo
    # O matplotlib ajusta automaticamente a escala para mostrar todos os dados
    max_time_real = max(times) if max_time is None else max_time
    plt.xlim(0, max_time_real)
    
    # Configurações de layout - garante que tudo seja capturado
    plt.tight_layout(pad=1.5)
    
    # Salva como string base64 - sem bbox_inches para capturar tudo
    buffer = io.BytesIO()
    plt.savefig(buffer, format='png', dpi=150, facecolor='white', edgecolor='none', pad_inches=0.2)
    buffer.seek(0)
    
    # Converte para base64
    image_base64 = base64.b64encode(buffer.getvalue()).decode()
    buffer.close()
    plt.close()
    
    return image_base64

def generate_all_charts(tests):
    """
    Gera todas as imagens dos gráficos
    
    Args:
        tests: Lista de testes (apenas os selecionados serão processados)
        
    Returns:
        Lista de tuplas (test_name, image_base64)
    """
    charts = []
    
    # Processa TODOS os testes selecionados (sem limitação)
    for test in tests:
        name = test.get('test_name', f"{t('threshold.test')} {test.get('id','')}")
        noise_data = test.get('noise_data', {})
        
        if isinstance(noise_data, dict) and noise_data:
            # Ordena os dados por tempo
            sorted_items = sorted(noise_data.items(), key=lambda x: float(x[0]))
            
            first_val = next(iter(noise_data.values()))
            if isinstance(first_val, dict) and 'value' in first_val:
                # Dados com estrutura completa
                times = [float(time_key) for time_key, _ in sorted_items]
                values = [data['value'] for _, data in sorted_items]
            else:
                # Dados simples
                times = [float(time_key) for time_key, _ in sorted_items]
                values = [value for _, value in sorted_items]
            
            if values and times and len(values) > 1:
                # Gera a imagem do gráfico
                image_base64 = generate_chart_image(times, values, name)
                if image_base64:
                    charts.append((name, image_base64))
    
    return charts

if __name__ == '__main__':
    # Teste
    from html_report_generator import _load_noise_history
    tests = _load_noise_history()
    charts = generate_all_charts(tests)
    print(f"Gerados {len(charts)} gráficos como imagens")
